 /**
 *
 * \file        HPRFGW_RFRec.c
 *
 * \brief       This file contains the RF receive task routines.		
 *
 * \detail		
 *
 * \author      Hazrat Shah	
 *
 * \date        08/22/2006
 * \warning	
 * \note
 * \review
 */

#include "hprfgw_config.h"
#if defined (OS_NUCLEUS) || defined (OS_WINCE)
	#include <string.h>
#endif
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfComLog.h"
#include "hprfgw_rfAppAspInterface.h"
#include "hprfgw_rf802_15_4.h"
#if defined (OS_NUCLEUS)
  #include "hprfgw_rfmc1319xHandler.h"
#endif
#include "hprfgw_rfIrqControlLib.h"
#include "hprfgw_rfResultCode.h"
#include "hprfgw_rfRetry.h"
#include "hprfgw_rfHeartbeat.h"
#include "hprfgw_rfRecMsg.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfMsgSystem.h"
#include "hprfgw_rfPlatformInit.h"
#ifndef FREESCALE_802_15_4_STACK
  #include "smac_802_15_4_shim.h"
  #include "smac_app.h"
#endif //FREESCALE_802_15_4_STACK
#include "smac_MC13192_regs.h"

#ifdef	RF_NETWORK_MASTER_DEVICE	    
#include "hprfgw_cresnet.h"
#endif
#include "smac_802_15_4_shim.h"

UINT32 Rfdiags_NumDataConfMessages=0;
UINT8  gPowerBoost = 0; // add this boost to the high power up to the limits: RF_PHY_POWER_LEVEL21

////////////////////////////////////////////////////////////////////////////////
//	DEFINITIONS	
////////////////////////////////////////////////////////////////////////////////
#ifndef RF_TASKS_BASE_PRIORITY
 #if defined (OS_NUCLEUS)
  #pragma info "!!!!Pass in the RF_TASKS_BASE_PRIORITY using the make file!!!!"
 #endif
#endif
#define RFREC_TASK_PRIORITY   					RF_TASK_PRIORITY(RF_TASKS_BASE_PRIORITY, 2)
#define	RFREC_TASK_STACK 						(1024*32)

#define	RFREC_PIPE_SIZE							(1024*8)
#define RFREC_PIPE_MAX_MSG_LEN					RF_DEVICE_DATA_HOLD_BUFFER_SIZE	

//	RF receiver event flags
//  Note: If changing this, make sure the bit numbers match the index below, for WinCE
#define	RFREC_MLM_PACKET_PENDING_EVENT			0x00000001	//	Event flag is set when MLM frame(s) are pending in the RF stack	 
#define	RFREC_MCP_PACKET_PENDING_EVENT			0x00000002	//	Event flag is set when MCP frame(s) are pending in the RF stack	 
#define	RFREC_ASP_PACKET_PENDING_EVENT			0x00000004	//	Event flag is set when ASP frame(s) are pending in the RF stack	 
#define	RFREC_RESULTCODE_TIMEOUT_EVENT			0x00000008	//	Event flag is set when any device result code timer has expired 	
#define	RFREC_RETRY_TIMEOUT_EVENT				0x00000010	//	Event flag is set when any device retry timer has expired 
#define RFREC_HEARTBEAT_TIMEOUT_EVENT			0x00000020	//	Event flag is set when any device heart beat timer has expired 

#if defined (OS_WINCE)
 // For WINCE, we need to create these seperate evetns for indexing 
 // Note: The indexing should match the bit number in the above defines for events
 //#define RFREC_MLM_PACKET_PENDING				0	//	Event flag is set when MLM frame(s) are pending in the RF stack	 
 //#define RFREC_MCP_PACKET_PENDING				1	//	Event flag is set when MCP frame(s) are pending in the RF stack	 
 //#define RFREC_ASP_PACKET_PENDING				2	//	Event flag is set when ASP frame(s) are pending in the RF stack	 
 //#define RFREC_RESULTCODE_TIMEOUT				3	//	Event flag is set when any device result code timer has expired 	
 //#define RFREC_RETRY_TIMEOUT					4	//	Event flag is set when any device retry timer has expired 
 //#define RFREC_HEARTBEAT_TIMEOUT				5	//	Event flag is set when any device heart beat timer has expired 

 //#define RFREC_EVENTS_COUNT					6   //  <<<--- Here specify the total number of events for this group

#endif

//	
//#define	STACK_DBG_RTE_TO_RTS_PATCH
//#define	STACK_DBG_STATUS_PATCH
//#define	RF_TEST

////////////////////////////////////////////////////////////////////////////////
//	LOCAL DATA		
////////////////////////////////////////////////////////////////////////////////
typedef struct	{
#if defined (OS_NUCLEUS)
    NU_TASK		task;
    UINT8   	stack[RFREC_TASK_STACK];
	
	NU_EVENT_GROUP events;	

	NU_PIPE  	pipe;
	UINT8 		pipebuf[RFREC_PIPE_SIZE];
#elif defined (OS_WINCE)
    HANDLE		task;
	
	//HANDLE	hEvents[RFREC_EVENTS_COUNT];
	HANDLE		hEvent;
	CRITICAL_SECTION RfRecCriticalSection;
	DWORD		RfRecEventGroup;

	HANDLE  	hReadPipe;
	HANDLE  	hWritePipe;
#elif defined (OS_LINUX)
    DM_RESOURCE_HANDLE	task;

	//HANDLE	hEvents[RFREC_EVENTS_COUNT];
    DM_RESOURCE_HANDLE		hEvents;
#if 0
	DM_RESOURCE_HANDLE RfRecCriticalSection;
#endif
	/* TODO: Should this be DWORD or something else? */
//	UINT32		RfRecEventGroup;
#endif

}	RFRECTASK;
RFRECTASK RFRecTask;

typedef	struct	{
	BOOL	uidvalid;				//	local device uid is valid
	UINT8	uid[UID_MAX_SIZE];		//	local device uid
	UINT16	panid;   				//	local device panid	
   	UINT16	shortaddr;				//	local device shortaddr
   	UINT8	addrmode;				//	local device addrmode	
	UINT8	channel;				//	local device channel	
	UINT16	power;					//	
	
	UINT8	CCAEnergyDetectThr;		//	CCA Energy detect threshold
	UINT8	RSSIPowerCompOffset;	//	Power comp offset added to RSSI level
	
	UINT8	security;				//	

	UINT8	retry;				
	UINT8	retryonconnectionreq;
	UINT8   retrydelay[RF_MAX_NO_OF_RETRIES_DELAYS];

	UINT16	resultcodetimeout;
	UINT8	heartbeattimeout;	
	UINT8	RFlinkLQIThr;			//	slave device link request LQI threshold  

	UINT8	PreLinkQueryPacketCnt;
}	TYPE_LOCAL_RF_DEVICE_COM_SETTINGS;
TYPE_LOCAL_RF_DEVICE_COM_SETTINGS LocalDevComSettings;

//	Gateway/Control system connection status
typedef	struct	{
	BOOL	gatewayconnected;
	BOOL	controlsystemconnected;
	BOOL	cresnettype;
	BOOL	ethernettype;
}	TYPE_GW_CS_CONNECTION_STATUS;
TYPE_GW_CS_CONNECTION_STATUS ConnStatus;

#ifdef FREESCALE_802_15_4_STACK
//	Mlm/MCP/Asp application input queues 
anchor_t mMlmeNwkInputQueue;
anchor_t mMcpsNwkInputQueue;
anchor_t mAspNwkInputQueue;
#endif

BOOL RFStackinitilizationComplete = FALSE;


const   UINT8   GATEWAY_DEFAULT_UID[UID_MAX_SIZE] = {0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11}; // Default UID
const   UINT8   GATEWAY_DEFAULT_RETRDEL[RF_MAX_NO_OF_RETRIES_DELAYS] = 
{
  GATEWAY_DEFAULT_RETRY_1_DELAY,
  GATEWAY_DEFAULT_RETRY_2_DELAY,
  GATEWAY_DEFAULT_RETRY_3_DELAY,
  GATEWAY_DEFAULT_RETRY_4_DELAY,
  GATEWAY_DEFAULT_RETRY_5_DELAY,
  GATEWAY_DEFAULT_RETRY_6_DELAY,
  GATEWAY_DEFAULT_RETRY_7_DELAY,
  GATEWAY_DEFAULT_RETRY_8_DELAY,
  GATEWAY_DEFAULT_RETRY_9_DELAY,
  GATEWAY_DEFAULT_RETRY_10_DELAY
};

const   UINT8   SLAVE_DEVICE_DEFAULT_UID[UID_MAX_SIZE] = {0x22,0x22,0x22,0x22,0x22,0x22,0x22,0x22}; // Default UID
const   UINT8   SLAVE_DEVICE_DEFAULT_RETRDEL[RF_MAX_NO_OF_RETRIES_DELAYS] = 
{
  SLAVE_DEVICE_DEFAULT_RETRY_1_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_2_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_3_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_4_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_5_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_6_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_7_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_8_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_9_DELAY,
  SLAVE_DEVICE_DEFAULT_RETRY_10_DELAY
};

#define	RF_POWER_AMP_GAIN	20	//	20dBm


const float RFTransmitPowerValuesInDbm[22] =	{
    //	dBm value,	dBm value,	power level 
    //	Out with    at 
    //	Gain		transceiver	


    ///////////////////////////////////////////////////
    //////  Table for new RF module with MC13202 chip
    ///////////////////////////////////////////////////
        (const float)-2.25, //  0
        (const float)5.0,   //  1
        (const float)9.0,   //  2
        (const float)12.1,  //  3
        (const float)12.6,  //  4
        (const float)13.1,  //  5
        (const float)13.7,  //  6
        (const float)19.5,  //  7
        (const float)20.2,  //  8
        (const float)20.44, //  9
        (const float)20.44, //  10
        (const float)22.3,  //  11
        (const float)22.3,  //  12
        (const float)22.31, //  13
        (const float)22.31, //  14
        (const float)22.31, //  15
        (const float)22.31, //  16	
        (const float)22.31, //  17
        (const float)22.31, //  18
        (const float)22.31, //  19
        (const float)22.31, //  20
        (const float)22.31  //  21
};

const float RFTransmitPowerValuesInDbmMC13192[22] =	{
    //	dBm value,	dBm value,	power level 
    //	Out with    at 
    //	Gain		transceiver	

		///////////////////////////////////////////////////
		//////  Table for old RF modules with MC13192 chip
		///////////////////////////////////////////////////
		(const float)-27.6+RF_POWER_AMP_GAIN,   //  0
		(const float)-20.6+RF_POWER_AMP_GAIN,   //  1
		(const float)-17.7+RF_POWER_AMP_GAIN,   //  2
		(const float)-16.3+RF_POWER_AMP_GAIN,   //  3
		(const float)-15.7+RF_POWER_AMP_GAIN,   //  4
		(const float)-15.2+RF_POWER_AMP_GAIN,   //  5
		(const float)-14.6+RF_POWER_AMP_GAIN,   //  6
		(const float)-8.9+RF_POWER_AMP_GAIN,    //  7
		(const float)-8.2+RF_POWER_AMP_GAIN,    //  8
		(const float)-7.5+RF_POWER_AMP_GAIN,    //  9
		(const float)-7.1+RF_POWER_AMP_GAIN,    //  10
		(const float)-1.6+RF_POWER_AMP_GAIN,    //  11
		(const float)-1.1+RF_POWER_AMP_GAIN,    //  12
		(const float)-0.7+RF_POWER_AMP_GAIN,    //  13
		(const float)-0.3+RF_POWER_AMP_GAIN,    //  14
		(const float)1.3+RF_POWER_AMP_GAIN,     //  15
		(const float)1.9+RF_POWER_AMP_GAIN,     //  16	
		(const float)2.5+RF_POWER_AMP_GAIN,     //  17
		(const float)2.6+RF_POWER_AMP_GAIN,     //  18
		(const float)3.2+RF_POWER_AMP_GAIN,     //  19
		(const float)3.7+RF_POWER_AMP_GAIN,     //  20
		(const float)4.1+RF_POWER_AMP_GAIN      //  21
};

//	rf power limit lookup table

TYPE_RF_POWER_LIMITS RFPowerLmts[][27]	=	{
        //	low                     high                        rf channels
        //	limits	                limits

    ///////////////////////////////////////////////////
    //////  Table for international version
    ///////////////////////////////////////////////////
    {
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	0	
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	1
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	2
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	3
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	4
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	5
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	6
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	7
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	8
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	9
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0		},		//	10
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	11
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	12
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	13
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	14	
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	15
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	16
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	17
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	18
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	19
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	20
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	21
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	22
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	23
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	24
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	25
    	{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2		},		//	26
    },
    ///////////////////////////////////////////////////
    //////  Table for new RF module with MC13202 chip
    ///////////////////////////////////////////////////
    {
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	0	
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	1
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	2
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	3
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	4
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	5
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	6
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	7
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	8
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	9
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	10
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL21    },  //	11
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	12
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	13
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	14	
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	15
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	16
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	17
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	18
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	19
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	20
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	21
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	22
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	23
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL10    },  //	24
        {	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL6     },  //	25
        {	RF_PHY_POWER_LEVEL0,	RF_PHY_POWER_LEVEL0     },  //	26
    }
};

TYPE_RF_POWER_LIMITS RFPowerLmtsMC13192[][27]	=	{
        //	low                     high                        rf channels
        //	limits	                limits
	{
		///////////////////////////////////////////////////
		//////  Table for old RF modules with MC13192 chip international version
		///////////////////////////////////////////////////
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  0
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  1
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  2
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  3
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  4
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  5
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  6
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  7
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  8
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  9
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  10
		{	RF_PHY_POWER_LEVEL1,    RF_PHY_POWER_LEVEL2     },  //  11
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  12
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  13
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  14	
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  15
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  16
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  17
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  18
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  19
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  20
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  21
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  22
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  23
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  24
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  25
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  26
	},
	{
		///////////////////////////////////////////////////
		//////  Table for old RF modules with MC13192 chip US version
		///////////////////////////////////////////////////
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  0
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  1
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  2
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  3
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  4
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  5
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  6
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  7
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  8
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  9
		{	RF_PHY_POWER_LEVEL0,    RF_PHY_POWER_LEVEL0     },  //  10
		{	RF_PHY_POWER_LEVEL3,    RF_PHY_POWER_LEVEL12    },  //  11
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  12
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  13
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  14	
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  15
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  16
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  17
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  18
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  19
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  20
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  21
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  22
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL16    },  //  23
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL10    },  //  24
		{	RF_PHY_POWER_LEVEL3,	RF_PHY_POWER_LEVEL6     },  //  25
		{	RF_PHY_POWER_LEVEL1,	RF_PHY_POWER_LEVEL2     },  //  26
	}
};

////////////////////////////////////////////////////////////////////////////////
//	EXTERNAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////
extern void   EnableInterrupt(BOOL hi, UINT32 mask);
extern void   DisableInterrupt(BOOL hi, UINT32 mask);
extern void	  IrqControlLib_DisableMC1319xIrq(void);

////////////////////////////////////////////////////////////////////////////////
//	LOCAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////
void RFReceive_ComSettingsInit (void);
UINT32 Rfdiags_getNumDataConfMessages(void) { return Rfdiags_NumDataConfMessages; }


#ifdef FREESCALE_802_15_4_STACK
// MNT - 4/26/2007 -  Only the Freescale Stack needs this
/******************************************************************************
* The following functions are called by the MAC to put messages into the
* Application's queue. They need to be defined even if they are not used
* in order to avoid linker errors.
******************************************************************************/
UINT8 MLME_NWK_SapHandler(nwkMessage_t * pMsg)	{
  /* Put the incoming MLME message in the applications input queue. */
  MSG_Queue(&mMlmeNwkInputQueue, pMsg);
  return gSuccess_c;
}

UINT8 MCPS_NWK_SapHandler(mcpsToNwkMessage_t *pMsg)	{
  /* Put the incoming MCPS message in the applications input queue. */
  MSG_Queue(&mMcpsNwkInputQueue, pMsg);
  return gSuccess_c;
}

UINT8 ASP_APP_SapHandler(aspToAppMsg_t *pMsg)	{
  /* Put the incoming ASP message in the applications input queue. */
  MSG_Queue(&mAspNwkInputQueue, pMsg);
  return gSuccess_c;  
}
#endif // #if FREESCALE_802_15_4_STACK


#if defined (OS_WINCE)
/**
 *
 * \date        12-15-2009
 *
 * \author      PM
 *
 * \brief       Event group set
 * 
 * \detail      
 *
 *
 * \param	    void	
 *
 * \return      void
 *
 * \retval      
 *
 */
static void RFRecSetEvent(DWORD event)
{
	EnterCriticalSection(&RFRecTask.RfRecCriticalSection);
	RFRecTask.RfRecEventGroup |= event;
	LeaveCriticalSection(&RFRecTask.RfRecCriticalSection);
	SetEvent(RFRecTask.hEvent);
}
/**
 *
 * \date        12-15-2009
 *
 * \author      PM
 *
 * \brief       Clear Event group
 * 
 * \detail      
 *
 *
 * \param	    void	
 *
 * \return      void
 *
 * \retval      
 *
 */
static DWORD RFRecReadAndClearEvents(void)
{
	DWORD CurrentEvents;
	
	EnterCriticalSection(&RFRecTask.RfRecCriticalSection);
	
	CurrentEvents = RFRecTask.RfRecEventGroup;
	RFRecTask.RfRecEventGroup = 0;
	// Reset the events here
	ResetEvent(RFRecTask.hEvent);
	
	LeaveCriticalSection(&RFRecTask.RfRecCriticalSection);
	return (CurrentEvents);
}

#endif


/**
 *
 * \date        03-21-2007
 *
 * \author      MNT
 *
 * \brief       Non Processor dependent HISR code for the receive timer
 * 
 * \detail      
 *
 *
 * \param	    void	
 *
 * \return      void
 *
 * \retval      
 *
 */
void RFReceive_Timer_HISR_Function(void)
{
	#define	MAX_RF_MLM_MAIN_LOOP	20	
#ifdef FREESCALE_802_15_4_STACK
	UINT8 macStatus;
	UINT8 maxloop = MAX_RF_MLM_MAIN_LOOP;
#endif
	UINT32 eventflgs;


#ifdef FREESCALE_802_15_4_STACK
	do	{
   		// Call the MAC main function. 
    	macStatus = Mlme_Main();	
	}	while (macStatus && --maxloop);
#endif //#ifdef FREESCALE_802_15_4_STACK

    eventflgs = 0;
#if defined (OS_NUCLEUS)
	if	(MSG_Pending(&mMlmeNwkInputQueue))	{	
		//	Raise event flag for the MLM frames 
		eventflgs |= RFREC_MLM_PACKET_PENDING_EVENT;
	}
	if	(MSG_Pending(&mMcpsNwkInputQueue))	{
		//	Raise event flag for the MCP frames		
		eventflgs |= RFREC_MCP_PACKET_PENDING_EVENT;
	}
	if	(MSG_Pending(&mAspNwkInputQueue))	{
		//	Raise event flag for the ASP frames		
		eventflgs |= RFREC_ASP_PACKET_PENDING_EVENT;
	}	

	if (eventflgs)	{
		NU_Set_Events(&RFRecTask.events, eventflgs,  NU_OR);	
	}
#elif defined (OS_WINCE)
	if	(MSG_Pending(mMlmeNwkInputQueue))	{
		eventflgs |= RFREC_MLM_PACKET_PENDING_EVENT;
	}
	if	(MSG_Pending(mMcpsNwkInputQueue))	{
		eventflgs |= RFREC_MCP_PACKET_PENDING_EVENT;
	}
	if	(MSG_Pending(mAspNwkInputQueue))	{
		eventflgs |= RFREC_ASP_PACKET_PENDING_EVENT;
	}
	#if 0
	// Now set the event based on the flags
	if (eventflgs & RFREC_MLM_PACKET_PENDING_EVENT)
	{
		SetEvent(RFRecTask.hEvents[RFREC_MLM_PACKET_PENDING]);
	}
	if (eventflgs & RFREC_MCP_PACKET_PENDING_EVENT)
	{
		SetEvent(RFRecTask.hEvents[RFREC_MCP_PACKET_PENDING]);
	}
	if (eventflgs & RFREC_ASP_PACKET_PENDING_EVENT)
	{
		SetEvent(RFRecTask.hEvents[RFREC_ASP_PACKET_PENDING]);
	}
	#endif
	RFRecSetEvent(eventflgs);
#endif
    if(MSG_Pending(mMlmeNwkInputQueue))  {
        //  Raise event flag for the MLM frames
        eventflgs |= RFREC_MLM_PACKET_PENDING_EVENT;
    }
    if(MSG_Pending(mMcpsNwkInputQueue))  {
        //  Raise event flag for the MCP frames
        eventflgs |= RFREC_MCP_PACKET_PENDING_EVENT;
    }
    if(MSG_Pending(mAspNwkInputQueue))   {
        //  Raise event flag for the ASP frames
        eventflgs |= RFREC_ASP_PACKET_PENDING_EVENT;
    }

    if(eventflgs)  {
        OsSetEvent(RFRecTask.hEvents, eventflgs);
    }
}

/**
 *
 * \date        09-21-2006
 *
 * \author      Hazrat 
 *
 * \brief       Set channel used during freescale stack startup
 * 
 * \detail      
 *
 *
 * \param	    UINT8 channel 
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 *
 */
UINT16 RFReceive_SetChannel (UINT8 channel)	{	
	mlmeMessage_t Msg;
	UINT16 res = SUCCESS;

	Msg.msgType = gMlmeSetReq_c;
	Msg.msgData.setReq.pibAttribute      = gMPibLogicalChannel_c;
	Msg.msgData.setReq.pibAttributeValue = &channel;

	if(NWK_MLME_SapHandler((mlmeMessage_t*) &Msg) != gSuccess_c)	{
		RFIntHostApp.pF_ErrorHandler("RFREC: Set channel message failed");
		res = FAILURE;
	}	

	return (res);
}

/**
 *
 * \date        09-21-2006
 *
 * \author      Hazrat 
 *
 * \brief       Set stack to idle mode used during freescale stack startup
 * 
 * \detail      
 *
 * \param	    UINT8 value 
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 *
 */
UINT16 RFReceive_RxOnIdle (UINT8 value)	{	
	mlmeMessage_t Msg;
	UINT16 res = SUCCESS;

	Msg.msgType = gMlmeSetReq_c;
	Msg.msgData.setReq.pibAttribute      = gMPibRxOnWhenIdle_c;
	Msg.msgData.setReq.pibAttributeValue = &value;

	if(NWK_MLME_SapHandler((mlmeMessage_t*) &Msg) != gSuccess_c)	{
		RFIntHostApp.pF_ErrorHandler("RFREC: RxOnIdle message failed");
		res = FAILURE;
	}	

	return (res);
}

/**
 *
 * \date        09-21-2006
 *
 * \author      Hazrat 
 *
 * \brief       Set panid used during freescale stack startup
 * 
 * \detail      
 *
 *
 * \param	    UINT16 panid
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 *
 */
UINT16 RFReceive_SetPANID (UINT16 panid)	{	
	mlmeMessage_t Msg;
	UINT16 res = SUCCESS;
	panid = ConvUINTToLittleIndian (panid);

	Msg.msgType = gMlmeSetReq_c;
	Msg.msgData.setReq.pibAttribute      = gMPibPanId_c;
	Msg.msgData.setReq.pibAttributeValue = (UINT8*) &panid;

	if(NWK_MLME_SapHandler((mlmeMessage_t*) &Msg) != gSuccess_c)	{
		RFIntHostApp.pF_ErrorHandler("RFREC: Set panid message failed");
		res = FAILURE;
	}	

	return (res);
}

/**
 *
 * \date        09-21-2006
 *
 * \author      Hazrat 
 *
 * \brief       Preform energy scan used during freescale stack startup
 * 
 * \detail      
 *
 * \param	    void
 *
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 *
 */
#ifdef	STACK_FIX_1
UINT16 RFReceive_EnergyScan (void)	{	
	mlmeMessage_t *pMsg;
  	mlmeScanReq_t *pScanReq;
	UINT16 res = FAILURE;
	UINT32 ChannelMask = RFTRANS_SCAN_ALL_CHANNEL_MASK;

	if (pMsg = MSG_AllocType(mlmeMessage_t))	{
		/* This is a MLME-START.req command */
		pMsg->msgType = gMlmeScanReq_c;
		/* Create the Start request message data. */
		pScanReq = &pMsg->msgData.scanReq;
		/* gScanModeED_c, gScanModeActive_c, gScanModePassive_c, or gScanModeOrphan_c */
		pScanReq->scanType = gScanModeED_c;
	  
		/* ChannelsToScan & 0xFF - LSB, always 0x00 */
		pScanReq->scanChannels[0] = 0x00;	//	(uint8_t)((ChannelMask)     & 0xFF);
		/* ChannelsToScan>>8 & 0xFF  */
		pScanReq->scanChannels[1] = (uint8_t)((ChannelMask>>8)  & 0xFF);
		/* ChannelsToScan>>16 & 0xFF  */
		pScanReq->scanChannels[2] = (uint8_t)((ChannelMask>>16) & 0xFF);
		/* ChannelsToScan>>24 & 0xFF - MSB */
		pScanReq->scanChannels[3] = (uint8_t)((ChannelMask>>24) & 0xFF);
		
		/* Duration per channel 0-14 (dc). T[sec] = (16*960*((2^dc)+1))/1000000.
		   A scan duration of 3 on 16 channels approximately takes 2 secs. */
		pScanReq->scanDuration = 1;
		
		if(MSG_Send(NWK_MLME, pMsg) != gSuccess_c)	{	      	
			/* One or more parameters in the Start Request message were invalid. */
			RFIntHostApp.pF_ErrorHandler("RFREC: ED scan request message failed");
			res = FAILURE;
		}	else	{
			res = SUCCESS;
		}
	}

	return (res);
}
#endif

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function starts the pan	 			           
 *
 * \detail		
 * \date        08/23/2006
 *
 * \param	    None
 *
 * \return		None				
 *
 * \retval		
 */
UINT16  RFReceive_StartPAN (void)	{
	/* Message for the MLME will be allocated and attached to this pointer */
	mlmeMessage_t *pMsg;
	appToAspMsg_t ASPMsg;	
#ifdef	RF_NETWORK_MASTER_DEVICE
	/* Pointer which is used for easy access inside the allocated message */
	mlmeStartReq_t *pStartReq;
	/* Boolean value that will be written to the MAC PIB */
	UINT8	boolFlag;
#endif
	/* Return value from MSG_send - used for avoiding compiler warnings */
	UINT16	result = SUCCESS;	
	UINT8   uid[UID_MAX_SIZE];
	UINT16  addr;

	RFReceive_GetLocalDeviceUID(uid);
	
	/* Allocate a message for the MLME (We should check for NULL). */
	pMsg = MSG_AllocType(mlmeMessage_t);
  
  	if(pMsg != NULL)	{
		
		RFReceive_RxOnIdle(FALSE);

		//	Set rf power level
		ASPMsg.msgType = gAppAspSetPowerLevelReq_c;    
		ASPMsg.msgData.aspSetPowerLevelReq.powerLevel = RFReceive_TranslatePowerToPHYLimits(LocalDevComSettings.power);
		if (APP_ASP_SapHandler(&ASPMsg) != gSuccess_c)	{
			RFIntHostApp.pF_ErrorHandler("RFREC: rf power level could not be set"); 
			result = FAILURE;
		}
		
		/* Set-up MAC PIB attributes. Please note that Set, Get, and Reset messages are not freed by the MLME. */
    
		/* We must always set the short address to something 
	       else than 0xFFFF before starting a PAN. */	    
		pMsg->msgType = gMlmeSetReq_c;
	    pMsg->msgData.setReq.pibAttribute = gMPibShortAddress_c;
		//	The short address is needed in little indian format (LSB .. MSB)
	    addr = ConvUINTToLittleIndian (LocalDevComSettings.shortaddr); 
		pMsg->msgData.setReq.pibAttributeValue = (UINT8 *)&addr;  
		if (NWK_MLME_SapHandler(pMsg) != gSuccess_c)	{
			RFIntHostApp.pF_ErrorHandler("RFREC: rf short address could not be set"); 
			result = FAILURE;
		}
    
#ifdef	RF_NETWORK_MASTER_DEVICE	    
		/* Set association mode: Allow association only for the network coordinator */
	    pMsg->msgType = gMlmeSetReq_c;
	    pMsg->msgData.setReq.pibAttribute = gMPibAssociationPermit_c;
		boolFlag = TRUE;
		pMsg->msgData.setReq.pibAttributeValue = &boolFlag;
		if (MSG_Send(NWK_MLME, pMsg) != gSuccess_c)	{
			RFIntHostApp.pF_ErrorHandler("RFREC: association cmd failed"); 
			result = FAILURE;
		}

		/* This is a MLME-START.req command */
	    pMsg->msgType = gMlmeStartReq_c;	    
	    /* Create the Start request message data. */
	    pStartReq = &pMsg->msgData.startReq;
	    /* PAN ID - LSB, MSB (little indian). The example shows a PAN ID of 0xBEEF. */
	    pStartReq->panId[0]	= LocalDevComSettings.panid%256;
		pStartReq->panId[1] = LocalDevComSettings.panid/256;
	    /* Logical Channel - the default of 11 will be overridden */
	    pStartReq->logicalChannel = LocalDevComSettings.channel;
	    /* Beacon Order - 0xF = turn off beacons */
	    pStartReq->beaconOrder = 0x0F;  
	    /* Superframe Order - 0xF = turn off beacons */
	    pStartReq->superFrameOrder = 0x0F;
		/* Be a PAN coordinator */
	    pStartReq->panCoordinator = TRUE;
		/* Don't use battery life extension */
	    pStartReq->batteryLifeExt = FALSE;
	    /* This is not a Realignment command */
	    pStartReq->coordRealignment = FALSE;
	    /* Dont use security */
	    pStartReq->securityEnable = FALSE;	    
	    /* Send the Start request to the MLME. */
	    if(MSG_Send(NWK_MLME, pMsg) != gSuccess_c)	{	      	
			/* One or more parameters in the Start Request message were invalid. */
			RFIntHostApp.pF_ErrorHandler("RFREC: start request message failed");
			result = FAILURE;
	    }
#else
		RFReceive_RxOnIdle  (FALSE);
		RFReceive_SetChannel(RFReceive_GetLocalDeviceChannel());		
		RFReceive_SetPANID  (RFReceive_GetLocalDevicePanID());		
		RFReceive_RxOnIdle  (TRUE);

		MSG_Free(pMsg); 
#endif
	}	else	{
  		//	Allocation of a message buffer failed. 
		RFIntHostApp.pF_ErrorHandler("RFREC: message allocation failed in network PAN");
    	result = FAILURE;
  	}

	return (result);
}  


/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function initializes the stack and related parameters  			           
 *
 * \detail		
 * \date        11/08/2006
 *
 * \param	    None
 *
 * \return		None			
 *
 * \retval		
 */
void RFReceive_StackInitialize (void)	{
	UINT8 outputPower;
// Initialize the correct stack based on the stack we are using
// By default use SMAC
#ifdef FREESCALE_802_15_4_STACK


#ifdef	STACK_DBG_RTE_TO_RTS_PATCH	
	UINT32 *preg;
	UINT32 res;
	preg = (UINT32*)(MC1319xInterrupt + 0x1B0);
	*preg = 0x4E750000;
	preg  = &res;
#endif


#ifdef	STACK_DBG_STATUS_PATCH
	UINT32 *preg;
	UINT32 res;
	//	patch the status reg overwrite to NOP at the begining of the MC13192 interrupt
	preg = (UINT32*)(MC1319xInterrupt + 0x30);
	*preg = 0x4E714E71;
	preg += 1;
	*preg = 0x4E714E71;
	preg += 1;
	*preg &= 0x0000FFFF;
	*preg |= 0x4E710000;
	//	patch the status reg overwrite to NOP at the end of the MC13192 interrupt
	preg = (UINT32*)(MC1319xInterrupt + 0x1F8);
	*preg = 0x4E714E71;
	preg += 1;
	*preg = 0x4E714E71;
	preg += 1;
	*preg &= 0x0000FFFF;
	*preg |= 0x4E710000;
	preg  = &res;
#endif

	// Disable receiver timer interrupt
	RFReceive_DisableTimerInterrupt();
	// Disable the MC13192 external interrupts 
  	IrqControlLib_DisableMC1319xIrq();
	
	//	Prepare the MLM, MCP, ASP input queues anchors
  	OsQueueClear(mMlmeNwkInputQueue);
  	OsQueueClear(mMcpsNwkInputQueue);
  	OsQueueClear(mAspNwkInputQueue);
	//	Initialize the 802.15.4 stack
	Init_802_15_4(LocalDevComSettings.CCAEnergyDetectThr, LocalDevComSettings.RSSIPowerCompOffset, (UINT8*) &LocalDevComSettings.uid); 		

#else //#if FREESCALE_802_15_4_STACK

  // MNT - 5/7/2007 - Moved this from the main HPRFGW init to here
  // MNT - 5/4/2007 - Channel 26, Max Power

  outputPower = RFReceive_TranslatePowerToPHYLimits(LocalDevComSettings.power);

  RF_Intfc_Init(LocalDevComSettings.channel, 
                outputPower,
                (UINT8*) &LocalDevComSettings.uid,
                LocalDevComSettings.CCAEnergyDetectThr,
                LocalDevComSettings.RSSIPowerCompOffset);
  Init_Smac_802_15_4_Shim();


#endif //#if FREESCALE_802_15_4_STACK

}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function is called to initializ the MAC parameters 			           
 *
 * \detail		
 * \date        08/23/2006
 * \param	    void
 *
 * \return		void			
 *
 * \retval		
 */
void RFReceive_PANInitialize (void)	{
	#define	RFREC_PAN_START_STATE				0
	#define	RFREC_PAN_START_CONFIRM_STATE		1
	#define RFREC_INITIALIZATION_DONE_STATE		2

#ifdef	RF_NETWORK_MASTER_DEVICE	
	UINT8	state = RFREC_PAN_START_STATE;
	nwkMessage_t* pMsgIn = NULL;
	UINT8	type     = ~gNwkScanCnf_c;	
	UINT8	attempts = 0;
	UINT8	loop;

	while (state < RFREC_INITIALIZATION_DONE_STATE)	{
		switch (state)	{		
			case	RFREC_PAN_START_STATE:
			    if (SUCCESS == RFReceive_StartPAN())	{
			        /* If the Start request was sent successfully to the MLME, then goto Wait for confirm state. */
			        state = RFREC_PAN_START_CONFIRM_STATE;
					attempts = 0;
					type     = ~gNwkStartCnf_c;
			    }	else	{
					//	try to start the PAN again
					if (++attempts > 10)	{
						RFIntHostApp.pF_ErrorPrintf ("RFREC: rf stack initialization failed, pan start command failed");
						state = RFREC_INITIALIZATION_DONE_STATE;									
					}
				}					
				break;

			case	RFREC_PAN_START_CONFIRM_STATE:
			    //	Stay in this state until the Start confirm message arrives, and then goto the Listen state. 
			    if (type == gNwkStartCnf_c)	{
			    	//	Done	 				    	
					state = RFREC_INITIALIZATION_DONE_STATE;
					attempts = 0;
			    }	else	{
					//	Stay in this state for confirmation
					if (++attempts > 10)	{
						RFIntHostApp.pF_ErrorPrintf ("RFREC: rf stack initialization failed, pan start command confirmation failed");
						state = RFREC_INITIALIZATION_DONE_STATE;									
					}					
				}			
				break;			
		}
    	
		if (state != RFREC_INITIALIZATION_DONE_STATE)	{ 
			loop = 20;
#ifdef FREESCALE_802_15_4_STACK
			while (Mlme_Main() && loop);
#endif //#ifdef FREESCALE_802_15_4_STACK

			if(MSG_Pending(mMlmeNwkInputQueue)) {
			    pMsgIn = MSG_DeQueue(mMlmeNwkInputQueue);
				type   = pMsgIn->msgType;
				MSG_Free(pMsgIn);
			}	
			RFIntHostApp.pF_HwDelayMsec(100);		
		}	else	{
			RFStackinitilizationComplete = TRUE;
		}
	}
#else
	RFReceive_StartPAN();
	RFIntHostApp.pF_HwDelayMsec(100);
	RFStackinitilizationComplete = TRUE;
#endif
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief				 			           
 *
 * \detail		
 * \date        10/11/2006
 *
 * \param	    UINT32 event_group
 *
 * \return		void	 			
 *
 * \retval		
 */
void	RFReceive_MessageProcessing (UINT32 event_group)	{
	int maxnoofloops = 300;
	void   *pMsgIn;

	do	{
		//	Try to get a message from the MLM buffer
		pMsgIn = NULL;
		
		//	Check for network control commands
		if (event_group & RFREC_MLM_PACKET_PENDING_EVENT)	{ 
		    HPRF_DEBUG(DmConsolePrintf("%s\n", __FUNCTION__);)
			//	Disable timer interrupt
			RFReceive_DisableTimerInterrupt ();
#if defined (OS_NUCLEUS)
			if	(MSG_Pending(&mMlmeNwkInputQueue))	{
#elif defined (OS_WINCE)
			if	(MSG_Pending(mMlmeNwkInputQueue))	{
#endif
#if defined (OS_NUCLEUS)
				pMsgIn = MSG_DeQueue(&mMlmeNwkInputQueue);
#elif defined (OS_WINCE)
				pMsgIn = MSG_DeQueue(mMlmeNwkInputQueue);
#endif
			if(MSG_Pending(mMlmeNwkInputQueue)) {
			    pMsgIn = MSG_DeQueue(mMlmeNwkInputQueue);
			    HPRF_QUEUE_DEBUG(DmConsolePrintf("--- %s:%i <--- dequeued bytes from mlmeNwkInputQueue\n", \
		                __FUNCTION__, __LINE__);)

				RFReceive_EnableTimerInterrupt ();			    				    				    			

				if (SUCCESS != RFRecMsg_MLMEMessageProcessing(RFDesc_GetMLMEDescriptiveTableIndex(), (nwkMessage_t*) pMsgIn))	{
					//	Error, message was not processed 
				}				
				//	Release buffer
				RFReceive_DisableTimerInterrupt ();
				MSG_Free(pMsgIn);	
				RFReceive_EnableTimerInterrupt ();
			}	else	{
				//	There are no more messages 
				event_group &= ~RFREC_MLM_PACKET_PENDING_EVENT;
			}						    				    				    		  		
			RFReceive_EnableTimerInterrupt ();			    				    				    		  		
		}

		if (event_group & RFREC_ASP_PACKET_PENDING_EVENT)	{ 
			//	Disable timer interrupt
			RFReceive_DisableTimerInterrupt ();			    				    								
			//	Try to get message from ASP
#if defined (OS_NUCLEUS)
			if	(MSG_Pending(&mAspNwkInputQueue))	{
#elif defined (OS_WINCE)
			if	(MSG_Pending(mAspNwkInputQueue))	{
#endif
			if (MSG_Pending(mAspNwkInputQueue)) {
		       	// Get the message from MCPS 
#if defined (OS_NUCLEUS)
		        pMsgIn = MSG_DeQueue(&mAspNwkInputQueue);
#elif defined (OS_WINCE)
				pMsgIn = MSG_DeQueue(mAspNwkInputQueue);
#endif
			    pMsgIn = MSG_DeQueue(mAspNwkInputQueue);
			    HPRF_QUEUE_DEBUG(DmConsolePrintf("--- %s:%i <--- dequeued bytes from aspNwkInputQueue\n", \
                        __FUNCTION__, __LINE__);)

				//OsDequeueBytes(mAspNwkInputQueue, pMsgIn);

		        RFReceive_EnableTimerInterrupt ();
				// Process the asp message 
		        if (SUCCESS != RFRecMsg_ASPMessageProcessing(RFDesc_GetASPDescriptiveTableIndex(), (aspToAppMsg_t*) pMsgIn))	{
		        	//	Error, message was not processed
		        }
				//	There are no more messages
		        RFReceive_DisableTimerInterrupt ();
				MSG_Free(pMsgIn);
				RFReceive_EnableTimerInterrupt ();
		    }	else	{
				//	There are no more messages
				event_group &= ~RFREC_ASP_PACKET_PENDING_EVENT;
			}			
			RFReceive_EnableTimerInterrupt ();
		}
							
		//	Check for data commands
		if (event_group & RFREC_MCP_PACKET_PENDING_EVENT)	{ 
			//	Disable timer interrupt
			RFReceive_DisableTimerInterrupt ();
			//	Check if any MCP messages are pending
#if defined (OS_NUCLEUS)
			if	(MSG_Pending(&mMcpsNwkInputQueue))	{
#elif defined (OS_WINCE)
			if	(MSG_Pending(mMcpsNwkInputQueue))	{
#endif
			if (MSG_Pending(mMcpsNwkInputQueue)) {
		       	// Get the message from MCPS
#if defined (OS_NUCLEUS)
		        pMsgIn = MSG_DeQueue(&mMcpsNwkInputQueue);
#elif defined (OS_WINCE)
				pMsgIn = MSG_DeQueue(mMcpsNwkInputQueue);
#endif
			    pMsgIn = MSG_DeQueue(mMcpsNwkInputQueue);
			    HPRF_QUEUE_DEBUG(DmConsolePrintf("--- %s:%i <--- dequeued bytes from mlmeNwkInputQueue\n", \
                        __FUNCTION__, __LINE__);)
				RFReceive_EnableTimerInterrupt ();
				
				// Messages from the MCPS must always be freed. 
		        RFReceive_DisableTimerInterrupt ();
	            if (((mcpsToNwkMessage_t*) pMsgIn)->msgType == gMcpsDataCnf_c)
	            {
	              Rfdiags_NumDataConfMessages++;          
	            }
			
				// Process the data messages 
		        if (SUCCESS != RFRecMsg_MCPMessageProcessing((mcpsToNwkMessage_t*) pMsgIn))	{
					//	Error, message was not procssed
				}

				MSG_Free(pMsgIn);
				RFReceive_EnableTimerInterrupt ();
			}	else	{
				//	There are no more messages						
				event_group &= ~RFREC_MCP_PACKET_PENDING_EVENT;
			}		
			RFReceive_EnableTimerInterrupt ();
		}

	}	while ((pMsgIn != NULL) && (--maxnoofloops));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This task scans the buttons and updates the led states			           
 *
 * \detail		
 * \date        08/21/2006
 *
 * \param	    UINT32 argc, void * argv
 *
 * \return		void			
 *
 * \retval		
 */
#if defined (OS_NUCLEUS)
void RFReceive_Task (UINT32 argc, void * argv)	{
#elif defined (OS_WINCE)
DWORD RFReceive_Task (void* pContext)	{
#endif
void RFReceive_Task(UINT32 context) {
	UINT32 events;
    UINT32 eventMask = RFREC_RETRY_TIMEOUT_EVENT | RFREC_RESULTCODE_TIMEOUT_EVENT |
            RFREC_MLM_PACKET_PENDING_EVENT | RFREC_MCP_PACKET_PENDING_EVENT|
            RFREC_ASP_PACKET_PENDING_EVENT;

#ifdef RF_NETWORK_MASTER_DEVICE
	UINT8	index;  		
	UINT8	uid[UID_MAX_SIZE];
#endif

	RFReceive_PANInitialize ();	

	RFReceive_EnableTimerInterrupt ();			    				    				    			

	RFIntHostApp.pF_StartupWait(HPRF_STARTUP_REC_READY);

#ifdef		RF_NETWORK_MASTER_DEVICE
	//	(5/25/07 - HS) - Broadcast gateway start up message to all devices, the devices needs to reestablish link
	//if ()	send packet if UID is valid
		memset (uid, 0, UID_MAX_SIZE);
		//	Create a Temporary Descriptive entry in tbl	(the descriptive entry is released once the packet is transmitted)		
		if (SUCCESS == RFDesc_CreateEntryInDescriptiveTable (&index, uid, RFReceive_GetLocalDevicePanID(), RFReceive_GetLocalDeviceAddrMode(), RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS))	{			
			//	Post startup msg 
			 RFTransmit_BuildGatewayReStartPacketAndPostToPipeByIndex (index, TRUE, 0, 1);
		}
	//}
#endif

#if defined (OS_WINCE)	
	RFIntHostApp.pF_ErrorPrintf("\r\nRFReceive_Task: Running!");
#endif

#ifdef RF_NETWORK_MASTER_DEVICE
	eventMask |= RFREC_HEARTBEAT_TIMEOUT_EVENT;
#endif
	while(1)	{
#if defined (OS_NUCLEUS)
		if (NU_Retrieve_Events(&RFRecTask.events, 0xffffffff, NU_OR_CONSUME, &event_group, NU_SUSPEND) == NU_SUCCESS)	{
#elif defined (OS_WINCE)
		if (WAIT_OBJECT_0 == WaitForSingleObject(RFRecTask.hEvent, OS_SUSPEND))
		{
			event_group = RFRecReadAndClearEvents();
#endif
	    events = OsWaitForEventAndClear(RFRecTask.hEvents, eventMask);
	    if(events)
	    {
			if (events & RFREC_RETRY_TIMEOUT_EVENT)	{
				//	Retry timeout event
				RFRetry_TimeoutProcessing();
			}	
			
			if (events & RFREC_RESULTCODE_TIMEOUT_EVENT)	{
				//	Result code event
				RFResultCode_TimeoutProcessing();	
			}	

	#ifdef RF_NETWORK_MASTER_DEVICE		
			if (event_group & RFREC_HEARTBEAT_TIMEOUT_EVENT)	{				
				//	Heart Beat event
				RFHeartBeat_TimeoutProcessing();
			}	
	#endif
			
			if (events & (RFREC_MLM_PACKET_PENDING_EVENT|RFREC_MCP_PACKET_PENDING_EVENT|RFREC_ASP_PACKET_PENDING_EVENT))	{
				//	Message waiting event
				RFReceive_MessageProcessing(events);
			}
		}
		else
		{
#if defined (OS_NUCLEUS)
			RFIntHostApp.pF_ErrorPrintf("RFReceive_Task:: Fail wait");
#elif defined (OS_WINCE)
			RFIntHostApp.pF_ErrorPrintf("RFReceive_Task:: Fail wait: %d", GetLastError());
#endif
			DmConsolePrintf("RFReceive_Task:: Fail wait\n");
		}
	}
}				

////////////////////////////////////////////////////////////////////////////////
//	GLOBAL FUNCTIONS 
//////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function is called on startup to create the rf receive task           
 *
 * \detail		The RF receive task, pipe is created, ..... bla bla...
 * 
 * \date        08/22/2006
 * \param	    None
 *
 * \return		None			
 *
 * \retval		
 */
void RFReceive_Init (void)	{
	BOOL	uidvalid;

#if defined (OS_WINCE)
	MSGQUEUEOPTIONS sMsgQueueOpt;
#endif


	RFIntHostApp.pF_IsUIDValid(&uidvalid);

	HPRF_DEBUG(DmConsolePrintf("%s uidvalid = %i\n", __FUNCTION__, uidvalid);)

	memset(&ConnStatus, 0, sizeof(ConnStatus));

	if (uidvalid)	{

		RFComLog_DbgLogInit ();	

		//	Clear the stack initialization complete flag
		RFStackinitilizationComplete = FALSE;
		
		//	Initialize rf communication parameters
		RFReceive_ComSettingsInit();

		RFDesc_Init();

#if defined (OS_WINCE)
		// Since the timer interrupt uses this, it is better to initialize here.	
		InitializeCriticalSection(&RFRecTask.RfRecCriticalSection);
#endif

	  // MNT - 5/7/2007 - Interrupts should be set up by the stack initialize
	  // call above.
	#ifdef FREESCALE_802_15_4_STACK
		RFReceive_StackInterruptConfig ();		
	#endif // FREESCALE_802_15_4_STACK

		RFRetry_Init();
		RFResultCode_Init();
		RFHeartBeat_Init();

		RFReceive_StackInitialize ();
        RFReceive_TimerInitialize();
    #ifndef RF_NETWORK_MASTER_DEVICE
		RFIntHostApp.pF_InitRoamingDescriptiveTable();
    #endif

#if defined (OS_NUCLEUS)
		//	Create RF receive task pipe
		if (NU_Create_Pipe(&RFRecTask.pipe, "PRFREC", &RFRecTask.pipebuf, RFREC_PIPE_SIZE,
                    		NU_VARIABLE_SIZE, RFREC_PIPE_MAX_MSG_LEN, NU_FIFO) != NU_SUCCESS)	{
       		RFIntHostApp.pF_SystemError (FATAL_RF_INIT);    	
		}
		
		//	Create event group	 
		if	(NU_Create_Event_Group(&RFRecTask.events, "ERFREC") != NU_SUCCESS)	{
    		RFIntHostApp.pF_ErrorHandler("ERFREC: Could not create rf receive event group!");
		}	

		//	Create RF receive task 
		if(NU_Create_Task(&RFRecTask.task, "TRFREC", RFReceive_Task, 0, NU_NULL,
			RFRecTask.stack, sizeof(RFRecTask.stack), RFREC_TASK_PRIORITY, 0, NU_PREEMPT,
			NU_START ) != NU_SUCCESS)	{
			RFIntHostApp.pF_ErrorHandler("TRFREC: Could not create task!");
		}
#elif defined (OS_WINCE)
		//Create RF receive task pipe
		//Set options for the message queue - once for all the queues created below
	    sMsgQueueOpt.dwSize = sizeof(sMsgQueueOpt);		//Size of the structure in bytes
	    sMsgQueueOpt.dwFlags = 0;						    //Describes the behavior of the message queue
	    sMsgQueueOpt.dwMaxMessages = 0;					//No limit on number of messages to queue at any one time, set to zero
	    // Message size, Maximum number of bytes in each message
	    sMsgQueueOpt.cbMaxMessage = (RFREC_PIPE_MAX_MSG_LEN); 
	    
		// Write handle to queue.
	    sMsgQueueOpt.bReadAccess = FALSE;
		RFRecTask.hWritePipe = CreateMsgQueue(TEXT("PRFREC"),  &sMsgQueueOpt);
	    
		// Read handle to queue
	    sMsgQueueOpt.bReadAccess = TRUE;
		RFRecTask.hReadPipe = CreateMsgQueue(TEXT("PRFREC"),  &sMsgQueueOpt);
	  
		if (NULL == RFRecTask.hWritePipe || NULL == RFRecTask.hReadPipe)
	    {
			RFIntHostApp.pF_ErrorPrintf("TRFREC: Error create MsgQ");
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
	    } // Create pipe for MlmeNwkInputQueue

		#if 0 // This method not used
		/* Create events and store the handles in array */
		for (i = 0; i < RFREC_EVENTS_COUNT; i++)
		{
			// No security attributes, Auto-reset event object, Initial state is nonsignaled, Unnamed object
			RFRecTask.hEvents[i] = CreateEvent (NULL, FALSE, FALSE, NULL);
		}
		#endif
		// Moved above, before the timer initialization
		//InitializeCriticalSection(&RFRecTask.RfRecCriticalSection);

		// Clear the eventgroup group
		RFRecTask.RfRecEventGroup = 0;

		// Create the event
		RFRecTask.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
		if (RFRecTask.hEvent == NULL)
		{
			RFIntHostApp.pF_ErrorPrintf("TRFREC: Fail create event!");
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
		}

		//	Create RF receive task 
		RFRecTask.task = CreateThread(NULL, 0, RFReceive_Task, pHPRDevice, 0, NULL);
		if (RFRecTask.task == NULL)
		{
			RFIntHostApp.pF_ErrorPrintf("TRFREC: Fail create task!");
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
		}
		// Set thread priority
		CeSetThreadPriority(RFRecTask.task, RFREC_TASK_PRIORITY);
#endif
        //  Create event group
        RFRecTask.hEvents = OsCreateEventGroup();
        if(!RFRecTask.hEvents) {
            DmConsoleLevelPrintf(DM_ERROR_LEVEL_FATAL, "ERFREC: Could not create rf receive event group!");
        }

        /* Need to create the event group before TimerInitialize, since
         * TimerInitialize will try to trigger that event. */
        RFRecTask.task = OsCreateNamedAdvTaskPri(&RFReceive_Task, 0, 0, RFREC_TASK_PRIORITY, "TRFREC");
        if(!RFRecTask.task) {
            DmConsoleLevelPrintf(DM_ERROR_LEVEL_FATAL, "TRFREC: Could not create task!\n");
        }
    }
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function initializes the gateway communication settings  			           
 *
 * \detail		This parameter copies the gateway settings from the non-viotale record to RAM
 *				If the affected non-viotalie parameters are changed then the gateway should be reset
 *				or the RF stack initialized, all connection to the Rf devices will be lost the 
 *				RF devices needs to go through the acquire process. This is TRUE for gateway UID and 
 *				panid. 	
 *						
 * \date        08/28/2006
 * \param	    None
 *
 * \return		None			
 *
 * \retval		
 */
void RFReceive_ComSettingsInit (void)
{

#ifdef RF_NETWORK_MASTER_DEVICE	
	UINT16 i;
	UINT8  uid[UID_MAX_SIZE];
	UINT16 panid;

	const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
	pHF->pF_GetLocalDeviceUID_PANID (
    (BOOL*) &LocalDevComSettings.uidvalid, 
    (UINT8*) &LocalDevComSettings.uid, 
    (UINT16*) &LocalDevComSettings.panid, 
    (UINT8 *)GATEWAY_DEFAULT_UID, 
    GATEWAY_DEFAULT_PANID); 

	pHF->pF_GetLocalDeviceUID_PANID ((BOOL*) &LocalDevComSettings.uidvalid, (UINT8*) &LocalDevComSettings.uid, (UINT16*) &LocalDevComSettings.panid, (UINT8 *)SLAVE_DEVICE_DEFAULT_UID, SLAVE_DEVICE_DEFAULT_PANID); 
	pHF->pF_GetShortAddress((UINT16 *) &LocalDevComSettings.shortaddr, GATEWAY_SHORT_ADDR);
	pHF->pF_GetAddressMode((UINT8 *)  &LocalDevComSettings.addrmode, GATEWAY_ADDR_MODE);
	pHF->pF_GetChannel   	((UINT8 *)  &LocalDevComSettings.channel, GATEWAY_DEFAULT_CHANNEL);
	pHF->pF_GetPowerLevel	( &LocalDevComSettings.power, GATEWAY_DEFAULT_POWER);	
	pHF->pF_GetCCAEnergyDetectThr ((UINT8 *)  &LocalDevComSettings.CCAEnergyDetectThr, GATEWAY_DEFAULT_CCAENERGY_THRESHOLD);		
	pHF->pF_GetRSSIPowerCompOffset    ((UINT8 *)  &LocalDevComSettings.RSSIPowerCompOffset, GATEWAY_DEFAULT_RSSIPOWERCOMPOFFSET);
	pHF->pF_GetRetryCount		((UINT8 *)  &LocalDevComSettings.retry, GATEWAY_DEFAULT_PACKET_TX_RETRY_COUNT);
	pHF->pF_GetRetryCountOnConnectionReq ((UINT8 *)  &LocalDevComSettings.retryonconnectionreq, GATEWAY_DEFAULT_PACKET_TX_RETRY_COUNT_ON_CONNECT_REQ);
	pHF->pF_GetResultCodeTimeout (&LocalDevComSettings.resultcodetimeout, GATEWAY_DEFAULT_RESULT_CODE_TIMEOUT);
    pHF->pF_GetRetryDelay  ((UINT8 *) &LocalDevComSettings.retrydelay, RF_MAX_NO_OF_RETRIES_DELAYS, (UINT8*)GATEWAY_DEFAULT_RETRDEL);
	pHF->pF_GetRetryCount ((UINT8*)&LocalDevComSettings.retry, GATEWAY_DEFAULT_PACKET_TX_RETRY_COUNT);	
	pHF->pF_GetHeartBeatRate (&LocalDevComSettings.heartbeattimeout, GATEWAY_DEFAULT_HEART_BEAT_TIMEOUT);
	pHF->pF_GetPreLinkQueryPacketCnt (&LocalDevComSettings.PreLinkQueryPacketCnt, GATEWAY_DEFAULT_PRELINK_QUERY_PACKETS);
	pHF->pF_GetGatewayRoamingSupport(&g_roamingEnabled);
	pHF->pF_GetPowerBoost(&gPowerBoost);
#else
	//	Init local device parameters 
	const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
	
	(pHF->pF_GetLocalDeviceUID_PANID) ((BOOL*) &LocalDevComSettings.uidvalid, (UINT8*) &LocalDevComSettings.uid, (UINT16*) &LocalDevComSettings.panid, (UINT8 *)SLAVE_DEVICE_DEFAULT_UID, SLAVE_DEVICE_DEFAULT_PANID); 
	(pHF->pF_GetRetryCount) (&LocalDevComSettings.retry, SLAVE_DEVICE_DEFAULT_PACKET_TX_RETRY_COUNT);
	(pHF->pF_GetRetryCountOnConnectionReq) ((UINT8 *)  &LocalDevComSettings.retryonconnectionreq, SLAVE_DEVICE_DEFAULT_PACKET_TX_RETRY_COUNT_ON_CONNECT_REQ);
	(pHF->pF_GetRetryDelay) ((UINT8*)&LocalDevComSettings.retrydelay, RF_MAX_NO_OF_RETRIES_DELAYS, (UINT8 *)SLAVE_DEVICE_DEFAULT_RETRDEL);	
	(pHF->pF_GetChannel) ((UINT8*)&LocalDevComSettings.channel, SLAVE_DEVICE_DEFAULT_CHANNEL);	
	(pHF->pF_GetPowerLevel) (&LocalDevComSettings.power, SLAVE_DEVICE_DEFAULT_POWER);	
	(pHF->pF_GetCCAEnergyDetectThr) (&LocalDevComSettings.CCAEnergyDetectThr, SLAVE_DEVICE_DEFAULT_CCAENERGY_THRESHOLD);	
	(pHF->pF_GetRSSIPowerCompOffset) (&LocalDevComSettings.RSSIPowerCompOffset, SLAVE_DEVICE_DEFAULT_RSSIPowerCompOffset);
	(pHF->pF_GetResultCodeTimeout) (&LocalDevComSettings.resultcodetimeout, SLAVE_DEVICE_DEFAULT_RESULT_CODE_TIMEOUT);	
	(pHF->pF_GetHeartBeatRate)(&LocalDevComSettings.heartbeattimeout, SLAVE_DEVICE_DEFAULT_HEART_BEAT_REFRESH_RATE);
	(pHF->pF_GetAddressMode)(&LocalDevComSettings.addrmode, SLAVE_DEVICE_ADDR_MODE);
	(pHF->pF_GetShortAddress)(&LocalDevComSettings.shortaddr, SLAVE_DEVICE_SHORT_ADDR);
	(pHF->pF_GetLocalDeviceRFLinkLQIThr)(&LocalDevComSettings.RFlinkLQIThr, SLAVE_DEVICE_DEFAULT_RFLINK_LQI_THR);
	(pHF->pF_GetPreLinkQueryPacketCnt) (&LocalDevComSettings.PreLinkQueryPacketCnt, SLAVE_DEVICE_DEFAULT_PRELINK_QUERY_PACKETS);
#endif		
}
							 							 

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks if the RF Stack initialization is complete
 *
 * \detail		
 *						
 * \date        09/01/2006
 *					
 * \param	    None
 *
 * \return		BOOL RFStackinitilizationComplete					
 *
 * \retval		TRUE/FALSE
 */
BOOL RFReceive_IsRFStackInitialized (void)	{	
	return (RFStackinitilizationComplete);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the local device UID   	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT8 *puid
 *
 * \return		None					
 *
 * \retval		None	
 */
void RFReceive_GetLocalDeviceUID (UINT8 *puid)	{	
	memcpy (puid, (UINT8*) &LocalDevComSettings.uid, UID_MAX_SIZE);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the CCA level    	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None	
 *
 * \return		CCA energy threshold 					
 *
 * \retval		None	
 */
UINT8 RFReceive_GetLocalDeviceCCAEnergyThr (void)	{	
	return (LocalDevComSettings.CCAEnergyDetectThr);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the RSSI power compensation threshold     	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None	
 *
 * \return		CCA energy threshold 					
 *
 * \retval		None	
 */
UINT8 RFReceive_GetLocalDeviceRSSIPowerCompThr (void)	{	
	return (LocalDevComSettings.RSSIPowerCompOffset);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the local device UID	   	     			           
 *			
 *
 * \detail		
 *						
 * \date        11/28/2006
 *					
 * \param	    UINT8 *puid
 *
 * \return		UINT16 res					
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16	RFReceive_SetLocalDeviceUID (UINT8 *puid)	{		
	UINT16 res = SUCCESS;
#ifdef	RF_NETWORK_MASTER_DEVICE
	INT8  buf[UID_MAX_SIZE*3];
#endif
	const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;

#ifdef	RF_NETWORK_MASTER_DEVICE
	res = (UINT16)pHF->pF_StoreLocalDeviceUID(puid);
#else
	res = (UINT16)pHF->pF_StoreLocalDeviceUID(puid);
#endif

	memcpy ((UINT8*) &LocalDevComSettings.uid, puid, UID_MAX_SIZE);

#ifdef	RF_NETWORK_MASTER_DEVICE
	ConvertUIDtoString (puid, buf, gAddrModeLong_c);
	SetCresnetSerialSubscriptionData (SERIAL_JOIN_UID, (UINT8*)buf, strlen(buf), TRUE);	
#endif

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets whether the local device UID is valid	   	     			           
 *			
 *
 * \detail		
 *						
 * \date        11/28/2006
 *					
 * \param	    BOOL Valid
 *
 * \return		UINT16 res 					
 *
 * \retval		SUCCESS/FAILRE		
 */
UINT16	RFReceive_SetLocalDeviceUIDValid (BOOL Valid)	{	
	UINT16 res = SUCCESS;
	LocalDevComSettings.uidvalid = Valid;	

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the device panid   	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    void
 *
 * \return		UINT16 panid					
 *
 * \retval		None	
 */
UINT16 RFReceive_GetLocalDevicePanID (void)	{		
	return (LocalDevComSettings.panid);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the device panid   	     			           
 *			
 *
 * \detail		
 *						
 * \date        11/30/2006
 *					
 * \param	    UINT16 panid
 *
 * \return		UINT16 res 					
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFReceive_SetLocalDevicePanID (UINT16 panid)	{		
	UINT16 res = SUCCESS;
	const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;

#ifdef	RF_NETWORK_MASTER_DEVICE
	res = (UINT16)pHF->pF_StoreLocalDevicePANID(panid);
#else
	res = (UINT16)pHF->pF_StoreLocalDevicePANID(panid);
#endif

	LocalDevComSettings.panid = panid;	

	if (RFReceive_IsRFStackInitialized())	{
		RFTransmit_BuildPANIDMessageAndPostToPipe (panid);	
	}

	return (res);
}

UINT8	RFReceive_GetLocalDeviceChannel (void)	{		
	return(LocalDevComSettings.channel);	 
}

UINT16	RFReceive_SetLocalDeviceChannel (UINT8 Channel)	{	
	UINT16 res = FAILURE;

	if ((Channel >= gLogicalChannel11_c) && (Channel <= gLogicalChannel26_c))	{ 
		res = SUCCESS;

		RFIntHostApp.pF_StoreChannel(Channel);

  		//	update channel  	
		LocalDevComSettings.channel = Channel; 
		
		if (RFReceive_IsRFStackInitialized())	{
			RFTransmit_BuildChannelSelectionFrameAndPostToPipe (Channel);
			//	Send the power level to the PHY, the channel level may be different
			RFTransmit_BuildPowerLevelFrameAndPostToPipe (LocalDevComSettings.power, TRUE);
		}		
	}
	
#ifdef	RF_NETWORK_MASTER_DEVICE
	//	report channel level to the control system
	SetCresnetAnalogSubscriptionData (ANALOG_JOIN_RF_CHANNEL, Channel, FALSE);
#endif

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the current power level	   	     			           
 *			
 *
 * \detail		 
 *						
 * \date        03/21/2007
 *					
 * \param	    None	
 *
 * \return		UINT16 power						
 *
 * \retval		power level index (high/low) 
 */
UINT16	RFReceive_GetLocalDevicePower (void)	{			
	return (LocalDevComSettings.power);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function looks up power level based on passed on index, 
 *				converts it to dBm and returns the power value in dBm    	   	     			           
 *			
 *
 * \detail		 
 *						
 * \date        03/21/2007
 *					
 * \param	    UINT16 power level index, BOOL Translate  	
 *
 * \return		float 						
 *
 * \retval		power level in dBm
 */
float RFReceive_ConvertPowerTodBm (UINT16 power, BOOL Translate)	{			

    float power_value;

    if (Translate)	{
		power = RFReceive_TranslatePowerToPHYLimits(power);
	}
	if (power > RF_PHY_MAX_POWER_LEVEL)	{	power = RF_PHY_MAX_POWER_LEVEL;	}
    
	if(RF_Intfc_GetVersion() < MC13202_CHIP_ID1)
		power_value = RFTransmitPowerValuesInDbmMC13192[power];		// Using older radio chip
	else
	    power_value = RFTransmitPowerValuesInDbm[power];

	return (power_value);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function looks up power level based on passed on index, 
 *				converts it to mWatts and returns the power value in mWatts  
 *
 * \detail		 
 *						
 * \date        03/21/2007
 *					
 * \param		UINT16 power, BOOL Translate	
 *
 * \return		float  						
 *
 * \retval		power in mwatts
 */
float RFReceive_ConvertPowerTomWatts (UINT16 power, BOOL Translate)	{			

    float power_value;
    float watts = 1;

    if (Translate)	{
		power = RFReceive_TranslatePowerToPHYLimits(power);
	}
	if (power > RF_PHY_MAX_POWER_LEVEL)	{	power = RF_PHY_MAX_POWER_LEVEL;	}

    if(RF_Intfc_GetVersion() < MC13202_CHIP_ID1)
		power_value = RFTransmitPowerValuesInDbmMC13192[power];		// Using older radio chip
	else
	    power_value = RFTransmitPowerValuesInDbm[power];

	//TODO: FIXME: Fix to compute watts properly.
	//watts = powf(10, (power_value)/10);
	return watts;
	//return (powf(10, (power_value)/10));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the power	   	     			           
 *			
 *
 * \detail		This function should only be called at run time by the host application
 *				On initialization the rf application will read the channel setting from the host
 *				The host does not need to call this function.
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT16 Power
 *
 * \return		UINT16 res					
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFReceive_SetLocalDevicePower (UINT16 Power)	{		
	UINT16 res = FAILURE;
	
	if (Power > RF_PHY_MAX_POWER_LEVEL)	{	Power = RF_PHY_MAX_POWER_LEVEL;	}

	res = RFIntHostApp.pF_StorePowerLevel(Power);
	
	LocalDevComSettings.power = Power;

	if (RFReceive_IsRFStackInitialized())	{
		RFTransmit_BuildPowerLevelFrameAndPostToPipe (Power, TRUE);
	}

#ifdef	RF_NETWORK_MASTER_DEVICE
	//	report power level to the control system
	SetCresnetAnalogSubscriptionData (ANALOG_JOIN_RF_POWER, Power, FALSE);
#endif

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the gateway short 16-bit addr 	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    void
 *
 * \return		UINT16 shortaddr					
 *
 * \retval		None	
 */
UINT16 RFReceive_GetLocalDeviceShortAddr (void)	{		
	return (LocalDevComSettings.shortaddr);
}

UINT16	RFReceive_SetLocalDeviceShortAddr (UINT16 shortaddr)	{		
	UINT16 res = SUCCESS;

  res = RFIntHostApp.pF_StoreShortAddress(shortaddr);

	LocalDevComSettings.shortaddr = shortaddr; 

	if (RFReceive_IsRFStackInitialized())	{
		RFTransmit_BuildShortAddrMessageAndPostToPipe (shortaddr);
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the gateway addr mode  (16-bit, 64-bit, none)	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    void
 *
 * \return		UINT8 addrmode					
 *
 * \retval		None	
 */
UINT8 RFReceive_GetLocalDeviceAddrMode (void)	{		
	return (LocalDevComSettings.addrmode);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the gateway addr mode  (16-bit, 64-bit, none)	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT8 addrmode
 *
 * \return		UINT16 res					
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFReceive_SetLocalDeviceAddrMode (UINT8 addrmode)	{		
	UINT16 res = SUCCESS;

  res = RFIntHostApp.pF_StoreAddressMode(addrmode);

	LocalDevComSettings.addrmode = addrmode;
	
	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the packet retry count  	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None	
 *
 * \return		UINT8 retry count					
 *
 * \retval		None	
 */
UINT8	RFReceive_GetLocalDeviceRetry (void)	{		
	return (LocalDevComSettings.retry);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the packet retry count  	     			           
 *			
 *
 * \detail		
 *						
 * \date        11/28/2006
 *					
 * \param	    UINT8 RetryCnt	
 *
 * \return		UINT16 res					
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16	RFReceive_SetLocalDeviceRetry (UINT8 RetryCnt)	{		
	UINT16 res = SUCCESS;

  res = RFIntHostApp.pF_StoreRetryCount(RetryCnt);

	LocalDevComSettings.retry = RetryCnt;
	
	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the packet retry count On Connection Request	 	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/30/2007
 *					
 * \param	    None	
 *
 * \return		UINT8 retry count On Connection Request				
 *
 * \retval		None	
 */
UINT8	RFReceive_GetLocalDeviceRetryOnConnectionReq (void)	{		
	return (LocalDevComSettings.retryonconnectionreq);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the packet retry count on connection request  	     			           
 *
 * \detail		
 *						
 * \date        10/30/2007
 *					
 * \param	    UINT8 RetryCnt	
 *
 * \return		UINT16 res					
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16	RFReceive_SetLocalDeviceRetryOnConnectionReq (UINT8 RetryCnt)	{		
	UINT16 res = SUCCESS;

	res = RFIntHostApp.pF_StoreRetryCountOnConnectionReq(RetryCnt);

	LocalDevComSettings.retryonconnectionreq = RetryCnt;
	
	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the packet retry delay time  	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT8 retry
 *
 * \return		UINT8 delay time					
 *
 * \retval		None	
 */
UINT8 RFReceive_GetLocalDeviceRetryDelayAtRetryCount (UINT8 retry)	{	
	UINT32 random;	

#ifdef	RF_NETWORK_MASTER_DEVICE		
	random = (RFAPP_GetRandomNum() & ((retry<2)?3:15));
#else
	random = (RFAPP_GetRandomNum() & 3);
#endif

	if (!random)	{	random = 1;	}	
	//if (retry >= RF_MAX_NO_OF_RETRIES_DELAYS)	{	retry = RF_MAX_NO_OF_RETRIES_DELAYS-1;	}
	//return (LocalDevComSettings.retrydelay[retry]);
	return ((UINT8) random);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the packet retry delays  	     			           
 *			
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT8 pbuf, UINT8 maxdlycount
 *
 * \return		UINT8 delaycount					
 *
 * \retval		None	
 */
UINT16 RFReceive_GetLocalDeviceRetryDelay (UINT8 *pRetryDly, UINT8 count, UINT8 index)	{			
	UINT16  res = SUCCESS;
	UINT8	i;

	if ((count + index) <= RF_MAX_NO_OF_RETRIES_DELAYS)	{	
		i = 0;
		while (count--)	{
			*(pRetryDly+i) = RFReceive_GetLocalDeviceRetryDelayAtRetryCount (index++);
			i++;
		}
	}	else	{
		res = FAILURE;
	}

	return (res);
}

UINT16	RFReceive_SetLocalDeviceRetryDelay (UINT8 *pRetryDly, UINT8 count, UINT8 index)	{
	UINT16 res = SUCCESS;
#if defined (RF_NETWORK_MASTER_DEVICE) || defined (RF_TEST)
	 UINT8  i, j;
#endif
	
	if ((count + index) <= RF_MAX_NO_OF_RETRIES_DELAYS)	{	
	#ifdef RF_NETWORK_MASTER_DEVICE 	
		for (i=0, j=index; i<count; i++, j++)	{
      if (RFIntHostApp.pF_StoreRetryDelay (*(pRetryDly+i), j) != SUCCESS)	{	res = FAILURE;	}
		}
	#endif

	#ifdef	RF_TEST 	
		for (i=0, j=index; i<count; i++, j++)	{
			if (RFIntHostApp.pF_StoreRetryDelay (*(pRetryDly+i), j) != SUCCESS)	{	res = FAILURE;	}
		}
	#endif
		
		memcpy((UINT8*)&LocalDevComSettings.retrydelay[index], pRetryDly, count);
	}	

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the packet result code timeout 	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None
 *
 * \return		None					
 *
 * \retval		None	
 */
UINT16 RFReceive_GetLocalDeviceResultCodeTimeout (void)	{		
	return (LocalDevComSettings.resultcodetimeout);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the packet result code timeout  	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT8 ResultCodeTimeout
 *
 * \return		UINT16 res 					
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16	RFReceive_SetLocalDeviceResultCodeTimeout (UINT16 ResultCodeTimeout)	{		
	UINT16 res = SUCCESS;

  res = RFIntHostApp.pF_StoreResultCodeTimeout(ResultCodeTimeout);
	
	LocalDevComSettings.resultcodetimeout = ResultCodeTimeout;

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the packet energy threshold	  	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT8 CCAEnergyDetectThr
 *
 * \return		UINT16 res						
 *
 * \retval		SUCCESS/FAILURE	 	
 */
UINT16	RFReceive_SetLocalDeviceCCAEnergyDetectThr (UINT8 CCAEnergyDetectThr)	{		
	UINT16 res = SUCCESS;

  res = RFIntHostApp.pF_StoreCCAEnergyDetectThr(CCAEnergyDetectThr);

	LocalDevComSettings.CCAEnergyDetectThr = CCAEnergyDetectThr;
	
	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reads the CCA energy threshold	  	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None
 *
 * \return		UINT8 offset						
 *
 * \retval		 	
 */
UINT8	RFReceive_GetLocalDeviceCCAEnergyDetectThr (void)	{		
	return (LocalDevComSettings.CCAEnergyDetectThr);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the RSSI power comp offset		  	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT8 RSSIPowerCompOffset
 *
 * \return		UINT16 res						
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16	RFReceive_SetLocalDeviceRSSIPowerCompOffset (UINT8 RSSIPowerCompOffset)	{		
	UINT16 res = SUCCESS;

  res = RFIntHostApp.pF_StoreRSSIPowerCompOffset(RSSIPowerCompOffset);

	LocalDevComSettings.RSSIPowerCompOffset = RSSIPowerCompOffset;
	
	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reads the RSSI power comp offset		  	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None
 *
 * \return		UINT8 offset						
 *
 * \retval				
 */
UINT8	RFReceive_GetLocalDeviceRSSIPowerCompOffset (void)	{
	return (LocalDevComSettings.RSSIPowerCompOffset);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function gets the heart beat timeout 	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None
 *
 * \return		None					
 *
 * \retval		None	
 */
UINT8 RFReceive_GetLocalDeviceHeartBeatTimeout (void)	{		
	return (LocalDevComSettings.heartbeattimeout);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the heart beat timeout 	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    UINT8 heartbeat
 *
 * \return		UINT16 res					
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFReceive_SetLocalDeviceHeartBeatTimeout (UINT8 heartbeat)	{		
	UINT16 res = SUCCESS;

  res = RFIntHostApp.pF_StoreHeartBeatRate(heartbeat);

	LocalDevComSettings.heartbeattimeout = heartbeat;
	
	return (res);
}	

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the link handler LQI threshold   	     			           
 *			
 *
 * \detail		
 *						
 * \date        09/25/2007
 *					
 * \param	    None
 *
 * \return		UINT8					
 *
 * \retval		LQI value 0..255	
 */
UINT8 RFReceive_GetLocalDeviceRFLinkLQIThr (void)	{		
	return (LocalDevComSettings.RFlinkLQIThr);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function saves the link handler LQI threshold  	     			           
 *			
 *
 * \detail		
 *						
 * \date        09/25/2007
 *					
 * \param	    UINT8 LQI
 *
 * \return		UINT16 res					
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFReceive_SetLocalDeviceRFLinkLQIThr (UINT8 LQI)	{		
	UINT16 res = SUCCESS;

	res = RFIntHostApp.pF_StoreLocalDeviceRFLinkLQIThr(LQI);
	LocalDevComSettings.RFlinkLQIThr = LQI;
	
	return (res);
}	

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the prelink query count  	     			           
 *			
 *
 * \detail		
 *						
 * \date        11/02/2007
 *					
 * \param	    None
 *
 * \return		UINT8					
 *
 * \retval		0..255	
 */
UINT8 RFReceive_GetLocalDevicePreLinkQueryPacketCnt (void)	{		
	return (LocalDevComSettings.PreLinkQueryPacketCnt);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function saves the prelink query count 	     			           
 *			
 *
 * \detail		
 *						
 * \date        11/02/2007
 *					
 * \param	    UINT8 PreLinkQuery count
 *
 * \return		UINT16 res					
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFReceive_SetLocalDevicePreLinkQueryPacketCnt (UINT8 Cnt)	{		
	UINT16 res = SUCCESS;

	res = RFIntHostApp.pF_StorePreLinkQueryPacketCnt(Cnt);
	LocalDevComSettings.PreLinkQueryPacketCnt = Cnt;
	
	return (res);
}	

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the result code timeout event	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None
 *
 * \return		None					
 *
 * \retval		None	
 */
void RFReceive_SetResultCodeTimeoutEvent (void)
{		
#if defined (OS_NUCLEUS)	
	NU_Set_Events(&RFRecTask.events, RFREC_RESULTCODE_TIMEOUT_EVENT,  NU_OR);
#elif defined (OS_WINCE)
	RFRecSetEvent(RFREC_RESULTCODE_TIMEOUT_EVENT);
#endif
	OsSetEvent(RFRecTask.hEvents, RFREC_RESULTCODE_TIMEOUT_EVENT);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the retry timeout event	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None
 *
 * \return		None					
 *
 * \retval		None	
 */
void RFReceive_SetRetryTimeoutEvent (void)
{		
#if defined (OS_NUCLEUS)
	NU_Set_Events(&RFRecTask.events, RFREC_RETRY_TIMEOUT_EVENT,  NU_OR);
#elif defined (OS_WINCE)
	RFRecSetEvent(RFREC_RETRY_TIMEOUT_EVENT);
#endif
	OsSetEvent(RFRecTask.hEvents, RFREC_RETRY_TIMEOUT_EVENT);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the heart beat timeout event	     			           
 *			
 *
 * \detail		
 *						
 * \date        10/10/2006
 *					
 * \param	    None
 *
 * \return		None					
 *
 * \retval		None	
 */
void RFReceive_SetHeartBeatTimeoutEvent (void)	
{
#if defined (OS_NUCLEUS)
	NU_Set_Events(&RFRecTask.events, RFREC_HEARTBEAT_TIMEOUT_EVENT,  NU_OR);
#elif defined (OS_WINCE)
	RFRecSetEvent(RFREC_HEARTBEAT_TIMEOUT_EVENT);
#endif
	OsSetEvent(RFRecTask.hEvents, RFREC_HEARTBEAT_TIMEOUT_EVENT);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns device security options 	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    None
 *
 * \return		UINT8					
 *
 * \retval		security option flags	
 */
UINT8 RFReceive_GetLocalDeviceSecurity (void)	{
	return (LocalDevComSettings.security);
}


///////////////////////////////////////////////////////////////////////////////////
//	Gateway/Control system connected status
///////////////////////////////////////////////////////////////////////////////////
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks whether the gateway is connected  	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    None
 *
 * \return		BOOL					
 *
 * \retval		Returns TRUE if gateway is connected 	
 */
BOOL	RFReceive_IsGatewayConnected (void)	{
	return (ConnStatus.gatewayconnected);      
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks whether the control system is connected   	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    None
 *
 * \return		BOOL					
 *
 * \retval		Returns TRUE if Ctl sys is connected 	
 */
BOOL	RFReceive_IsControlSystemConnected (void)	{
	return (ConnStatus.controlsystemconnected);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the gateway connection status   	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    BOOL
 *
 * \return		None					
 *
 * \retval		None	 	
 */
void	RFReceive_SetGatewayConnectStatus (BOOL Status)	{
	BOOL  isTraceEnabled = FALSE;
	
	HPRF_DEBUG(pr_info("===== %s ===== ConnStatus.gatewayconnected = %u Status = %u\n",\
	        __FUNCTION__, ConnStatus.gatewayconnected, Status);)
	if (ConnStatus.gatewayconnected != Status)	{
		ConnStatus.gatewayconnected = Status;

		if (ConnStatus.gatewayconnected == FALSE)	{
			//	On gateway connection lose also, clear control system connection status
			RFReceive_SetControlSystemConnectedStatus(FALSE, FALSE);	
			RFReceive_SetControlSystemConnectedNetTypeCresnet(FALSE);
			RFReceive_SetControlSystemConnectedNetTypeEthernet(FALSE);
		}

	#ifdef	RF_NETWORK_MASTER_DEVICE
		
	#else	
		//	Report gateway connect status change to the host application		
		RFIntHostApp.pF_GatewayConnectionStatusChange(Status);
		
		RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);
		if (isTraceEnabled)	{
			RFIntHostApp.pF_ConsolePrintf("Gateway connection status change: %s\r\n", Status?"Connected":"Disconnected");
		}
	#endif
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets Ctl Sys connection status   	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    BOOL	
 *
 * \return		None				
 *
 * \retval		 	
 */
void	RFReceive_SetControlSystemConnectedStatus (BOOL Status, BOOL SendUpdateReq)	{
	BOOL  isTraceEnabled = FALSE;
		
	if (ConnStatus.controlsystemconnected != Status)	{
		ConnStatus.controlsystemconnected = Status;	

		#ifdef	RF_NETWORK_MASTER_DEVICE

		#else	
			if (RFReceive_IsGatewayConnected() || (!ConnStatus.controlsystemconnected))	{
				//	Report CS status change to the host application		
				RFIntHostApp.pF_ControlSystemConnectionStatusChange(Status, SendUpdateReq);
			}	
			
			RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf("Control System connection status change: %s\r\n", Status?"Connected":"Disconnected");
			}
		#endif
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks whether ctl sys conneciton type is Cresnet    	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    None	
 *
 * \return		BOOL				
 *
 * \retval		Returns TRUE if Ctl sys connection type is Cresnet 	
 */
BOOL	RFReceive_IsControlSystemConnectedNetTypeCresnet (void)	{
	return (ConnStatus.cresnettype);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets ctl sys conneciton type     	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    BOOL
 *
 * \return		None				
 *
 * \retval		 	
 */
void	RFReceive_SetControlSystemConnectedNetTypeCresnet (BOOL Status)	{
	BOOL  isTraceEnabled = FALSE;
	
	if (ConnStatus.cresnettype != Status)	{
		ConnStatus.cresnettype = Status;

	#ifdef	RF_NETWORK_MASTER_DEVICE
		
	#else	
		RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);
		if (isTraceEnabled)	{
			RFIntHostApp.pF_ConsolePrintf("Control System connection type cresent: %s\r\n", Status?"Connected":"Disconnected");
		}

	#endif
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks whether ctl sys conneciton type is Ethernet     	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    None	
 *
 * \return		BOOL				
 *
 * \retval		Returns TRUE if Ctl sys connection type is Ethernet
 */
BOOL	RFReceive_IsControlSystemConnectedNetTypeEthernet(void)	{
	return (ConnStatus.ethernettype);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets ctl sys conneciton type     	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    BOOL
 *
 * \return		None				
 *
 * \retval		 	
 */
void	RFReceive_SetControlSystemConnectedNetTypeEthernet(BOOL Status)	{
	BOOL  isTraceEnabled = FALSE;
	
	if (ConnStatus.ethernettype != Status)	{
		ConnStatus.ethernettype = Status;
	#ifdef	RF_NETWORK_MASTER_DEVICE
		
	#else	
		RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);
		if (isTraceEnabled)	{
			RFIntHostApp.pF_ConsolePrintf("Control System connection type ethernet: %s\r\n", Status?"Connected":"Disconnected");
		}

	#endif
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function clears stack transmit buffers      	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    None	
 *
 * \return		UINT16				
 *
 * \retval		Returns SUCCESS if successful
 */
UINT16	RFReceiver_RFStackMsgQueInit (void)	{
	
	//RFIntHostApp.pF_ConsolePrintf ("Reseting the RF stack message queues \r\n");
	RFReceive_DisableTimerInterrupt();

	OsQueueReset(mMlmeNwkInputQueue);
	OsQueueReset(mMcpsNwkInputQueue);
	OsQueueReset(mAspNwkInputQueue);

  	RFReceive_EnableTimerInterrupt();

	return (SUCCESS);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		Get the total retry time in seconds   
 *
 * \detail			 
 *
 * \date        05/17/2007
 *
 * \param	    None		
 *
 * \return		UINT32 	 				
 *
 * \retval		Retry delay in seconds	 	  
 */	
UINT32	RFReceiver_GetTotalRetryDlyTimeInSeconds (void)	{
	UINT32 timeout = 0, i = 0;
	UINT32 retrycnt = RFReceive_GetLocalDeviceRetry();

	//	The retry delays are in steps of 100msec
	while (retrycnt > i)	{
		timeout += RFReceive_GetLocalDeviceRetryDelayAtRetryCount(i); 
		i++;
	}
	return ((timeout/10)+((timeout%10)?1:0));
}



/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function converts rf power level from a 2-bytes value to the RF PHY chip range      	     			           
 *			
 *
 * \detail		
 *						
 * \date        05/31/2007
 *					
 * \param	    UINT16 power	
 *
 * \return		UINT8				
 *
 * \retval		
 */
UINT8	RFReceive_TranslatePowerToPHYLimits(UINT16 power)	{
	UINT8 channel = RFReceive_GetLocalDeviceChannel();
    UINT8 retV, index;
    
    index = GetRFPowerTableIndex();

    if(RF_Intfc_GetVersion() < MC13202_CHIP_ID1)
		retV =  ((power)?RFPowerLmtsMC13192[index][channel].high:RFPowerLmtsMC13192[index][channel].low) + gPowerBoost;		// Using older radio chip
	else
		retV =  ((power)?RFPowerLmts[index][channel].high:RFPowerLmts[index][channel].low) + gPowerBoost;

    if( retV > RF_PHY_POWER_LEVEL21 )
		retV = RF_PHY_POWER_LEVEL21;

    //RETAILMSG(1, (L"RFReceive_TranslatePowerToPHYLimits: power = 0x%x, index = 0x%x, channel = 0x%x, retV = 0x%x\r\n", power, index, channel, retV));

	return retV;
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		return link quality in 2-byte range derived from %  
 *
 *
 * \detail			 
 *
 * \date        11/19/2007
 *
 * \param	    UINT8 linkQuality, INT8 offset		
 *
 * \return		UINT16	 	 				
 *
 * \retval		Returns link quality 	 	  
 */	
UINT8	RFReceive_AddLQIOffset (UINT8 linkQuality, INT8 offset)	{
		
	if (linkQuality)	{
		if (offset>=0)	{
			linkQuality = ((linkQuality+offset)>=255)?255:(linkQuality+(UINT8)offset);
		}	else	{
			offset *= -1;
			if (offset >= linkQuality)	{
				linkQuality = 0;	
			}	else	{
				linkQuality -= (UINT8)offset;
			}
		}
	}

	return (linkQuality);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		converts measured LQI value to 0..0xFF  
 *
 * \detail		returns values between -95 dBm to about -16.5 dBm which are represented by decimal
 *              values 190 (0xbe) and 33 respectively. 	 
 *				
 *				Conversion: 
 *				1)	clip incoming linkquality btw 190 to 33
 *				2)	Convert to 0..255 (based on IEEE recommendation)
 *				3)	LQI =  Linkquality + (Linkquality/2) + (Linkquality/8)
 *
 * \date        07/02/2007
 *
 * \param	    UINT8 linkQuality, INT8 offset		
 *
 * \return		UINT8  	 				
 *
 * \retval		Returns Link Quality level	 	  
 */	
UINT8	RFReceive_ConvertLinkQualityIEEEByte (UINT8 linkQuality, INT8 offset)	{
  if (linkQuality > 190) {
	linkQuality = 190; //190 = -95dBm as floor (translates to 0x00)
  }
  if (linkQuality < 33) {
	linkQuality = 33; //33 = -16.5 dBm as top (saturation) 
  }
  
  linkQuality = 190 - linkQuality;  
  linkQuality = linkQuality + (linkQuality >> 1) + (linkQuality >> 3);
  
  return (RFReceive_AddLQIOffset (linkQuality, offset));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		this function converts link quality to dBm  
 *
 *
 * \detail			 
 *
 * \date        07/02/2007
 *
 * \param	    UINT8 linkQuality		
 *
 * \return		INT8 	 				
 *
 * \retval		Returns link quality in mWatts	 	  
 */	
INT8	RFReceive_ConvertLinkQualityTodBm (UINT8 linkQuality)	{
	  if (linkQuality > 190) {
		linkQuality = 190; //190 = -95dBm as floor (translates to 0x00)
	  }
	  if (linkQuality < 33) {
		linkQuality = 33; //33 = -16.5 dBm as top (saturation) 
	  }
	  
	  linkQuality /= 2;

	  return ((INT8)(linkQuality*-1));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		this function converts link quality to mWatts 
 *
 *
 * \detail			 
 *
 * \date        07/02/2007
 *
 * \param	    UINT8 linkQuality		
 *
 * \return		float 	 				
 *
 * \retval		Returns link quality in mWatts	 	  
 */	
float	RFReceive_ConvertLinkQualityTomWatts(UINT8 linkQuality)	{
  //	return link quality in mWatts
  float lq = RFReceive_ConvertLinkQualityTodBm(linkQuality);
  float watts = 1;
  //TODO: FIXME: Convert to watts
  //return (powf(10, lq/10));
  return watts;
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		this function converts link quality to percent   
 *
 *
 * \detail			 
 *
 * \date        07/02/2007
 *
 * \param	    UINT8 linkQuality		
 *
 * \return		float  	 				
 *
 * \retval		Returns link quality 0...100%	 	  
 */	
UINT8 RFReceive_ConvertLinkQualityToPercent (UINT8 linkQuality, INT8 offset)	{
	//	note: the lower limit is set to 0 (this is the TRUE power percentage based on power level in mWatts)
	//return ((RFReceive_ConvertLinkQualityTomWatts(linkQuality)*100)/(RFReceive_ConvertLinkQualityTomWatts(33)));
	//	note: this method of percent conversion is based on IEEE proposed conversion scale

    //TODO: FIXME: Can't use floating point in the Linux kernel...
    //float result = RFReceive_ConvertLinkQualityToIEEEWord(linkQuality, offset) / (float)655.35;
    UINT8 milliPercent = RFReceive_ConvertLinkQualityToIEEEWord(linkQuality, offset) / 655;
	return milliPercent;
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		return link quality in 2-byte range derived from %  
 *
 *
 * \detail			 
 *
 * \date        07/02/2007
 *
 * \param	    UINT8 linkQuality, INT8 offset		
 *
 * \return		UINT16	 	 				
 *
 * \retval		Returns link quality from 0..65535	 	  
 */	
UINT16	RFReceive_ConvertLinkQualityToIEEEWord (UINT8 linkQuality, INT8 offset)	{
	return (RFReceive_ConvertLinkQualityIEEEByte(linkQuality, offset)*257);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		ED returns values between -95 dBm to about -18 dBm which are represented by decimal
 *				values 190 (0xbe) and 36 (0x24) respectively.
 *
 *
 * \detail			 
 *
 * \date        07/02/2007
 *
 * \param	    UINT8 energylevel		
 *
 * \return		UINT8 	 				
 *
 * \retval		Returns energy level 	 	  
 */	
UINT8	RFReceive_ConvertEnergyLevel(UINT8 energyLevel)	{
	//	Make dynamics of the energy level vary from 0x00-0xff
	if (energyLevel > 150) {
		energyLevel = 150; //150 = -75dBm as floor (translates to 0x00)
	}
	if (energyLevel < 23) {
		energyLevel = 23; //23 = -11.5 dBm as top (saturation)
	}
	energyLevel = 150 - energyLevel;  // 150 = -75dBm as floor (translates to 0xFF)
	energyLevel = energyLevel << 2;
	return (energyLevel);
}

//////////////////////////////////////////////////////////////////////////////////
////////////// Define check //////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////

#if defined (FREESCALE_802_15_4_STACK)
	#error "Windows CE dependcy needs to be worked on"
#endif

//////////////////////////////////////////////////////////////////////////////////
